﻿/*	Version 1.1
1.1		Case is completely ignored.  Strings are always compared as lower-case.


USAGE: 
	#include "functions/fuzzySearch.as"
	var searchThese_ary = [
		"ab", 
		"abc", 
		"abcdef", 
		"bcdefg", 
		"bcde", 
		"1ab", 
		"2", 
		12, 
		2, 
		1, 
		9, 
		6, 
		96, 
		196, 
		"bc-d", 
		"b-cd", 
		"b-c-d", 
		"-b-c-d", 
		false, 
		true, 
		undefined, 
		null, 
		(1/0)
	];
	var findThis_str = "bcd";		// bcd
	var index_ary = fuzzySearch( findThis_str, searchThese_ary );
	
	var bestMatch_str = searchThese_ary[ index_ary[0] ];
	trace("bestMatch_str: "+bestMatch_str);
	
	var result_ary = new Array( searchThese_ary.length );
	for(var i=0; i<searchThese_ary.length; i++)			result_ary[ i ] = searchThese_ary[ index_ary[i] ];
	trace( result_ary );


DESCRIPTION: 
	fuzzySearch() takes a string and an array.  It searches for the string within the array of strings  (it'll treat every item as a string,  reading undefined as the word "undefined")
	fuzzySearch() returns an index-array.  The 1st item of it indicates the location of the best-match within the input-array.  ( output[0] contains the index-location of the best-matching item in the input-array )
	
NOTES: 
	exact-matches take priority over approximate matches
	exact-matches with fewer total characters take priority over exact-matches with extra characters  (prefixes and suffixes are penalized)
	approximate matches with fewer gaps between matching letters take priority over approximate matches with more gaps  (a "gap" is any non-matching character.  Such as searching for "bcd" within "bc-d")
*/



function fuzzySearch( findThis_str, inThis_ary ){
	if( !findThis_str )					return undefined;
	if( !inThis_ary )						return undefined;
	if( !inThis_ary.length )		return undefined;
	
	var fullMatchBoost = 1000;  // always higher priority than findThis_str-matches  (assumes that findThis_str matches will be less than 1000 chars)
	var gapPenalty = 0.1;		// (0.1 recommended)		searching for "bcd"		0.1-0.4 => b-c-d before abc		0.5-0.9 => abc before b-c-d		1.0 => each gap cancels-out a matching letter
	
	// inputs:  findThis_str, a, b
	var index_ary = inThis_ary.sort( function( a, b ){
		var a_fullItem = String( a ).toLowerCase();
		var b_fullItem = String( b ).toLowerCase();
		var a_score = 0;
		var b_score = 0;
		var a_isFullMatch = (a_fullItem.indexOf(findThis_str) > -1);
		var b_isFullMatch = (a_fullItem.indexOf(findThis_str) > -1);
		// calculate match scores
		if( a_isFullMatch )		a_score += fullMatchBoost;
		else	a_score += getMatchScore( a_fullItem );
		if( b_isFullMatch )		b_score += fullMatchBoost;
		else	b_score += getMatchScore( b_fullItem );
		
		if(a_score > b_score)		return -1;		// bigger scores go higher on the list
		if(a_score < b_score)		return 1;			// smaller scores go lower on the list
		// if both options match,  then pick the shorter option
		if(a_score === b_score){
			if( a_fullItem.length < b_fullItem.length )		return -1;		// shorter fullItem lengths go higher on the list
			if( a_fullItem.length > b_fullItem.length )		return 1;			// longer fullItem lengths go shorter on the list
		}
		// otherwise, do not change sorting
		return 0;
	},  Array.RETURNINDEXEDARRAY  );// sort()
	
	
	// figure out how many letters of the findThis_str string are in the fullItem
	function getMatchScore( fullItem ){
		var score = 0;
		var lastFoundAt = -1;
		var startAt = 0;
		for( var i = 0;  i < findThis_str.length;  i++ ){
			var thisChar = findThis_str.charAt( i );
			var foundAt = fullItem.indexOf( thisChar, startAt );
			if( foundAt > -1 ){
				// count the number of mis-matched characters between matched characters and penalize the score
				if( lastFoundAt !== -1 )
				{// if a previous match was found
					var gaps = (foundAt - lastFoundAt) -1;		// -1 because a difference of 1 in the index indicates consecutive characters,  which should not be penalized
					score -= ( gaps * gapPenalty );
				}// if a previous match was found
				lastFoundAt = foundAt;
				// findThis_str matches fullItem more than before
				score++;
				// only consider the parts of fullItem after this match  (ignore everything before this match from now on)  (search for letters in order)
				startAt = foundAt+1;
			}
		}// for:  each character of the findThis_str string
		return score;
	}// getMatchScore()

	
	return index_ary;
	
}// fuzzySearch()